/* $Id: arbsrct.c,v 1.4 1998/09/16 02:04:56 ericb Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Arb source test program to verify source mode/srcbuffer
   mode/srcbuffer size/xfer size combinations */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	if (error_print)\
	    (void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	if (error_print)\
	    (void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: arbsrct.c,v 1.4 1998/09/16 02:04:56 ericb Exp $";
static const char *progname;

static int error_print;

static int
check_src_status(E1432ID hw, int nchan, SHORTSIZ16 *chan_list,
		 int overread_ok)
{
    int     i, status;

    for (i = 0; i < nchan; i++)
    {
	/* For these functions, negative return value means error,
	   positive means the condition happenned.  Either way, we
	   want to know about it. */

	status = e1432_check_src_overload(hw, chan_list[i]);
	if (status != 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_check_src_overload: returned %d\n",
			   progname, status);
	    return -1;
	}

	status = e1432_check_src_overread(hw, chan_list[i]);
	if (status != 0 && !overread_ok)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_check_src_overread: returned %d\n",
			   progname, status);
	    return -1;
	}

	status = e1432_check_src_shutdown(hw, chan_list[i]);
	if (status != 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_check_src_shutdown: returned %d\n",
			   progname, status);
	    return -1;
	}

	status = e1432_check_src_underrun(hw, chan_list[i]);
	if (status != 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_check_src_underrun: returned %d\n",
			   progname, status);
	    return -1;
	}
    }

    return 0;
}

static int
check_arbrdy(E1432ID hw, int nchan, SHORTSIZ16 *chan_list, int dmode)
{
    int     i, status;

    for (i = 0; i < nchan; i++)
    {
	CHECK(status = e1432_check_src_arbrdy(hw, chan_list[i], dmode));
	if (status == 0)
	    return 0;
    }
    return 1;
}

static int
check_arbrdy2(E1432ID hw, int nchan, SHORTSIZ16 *chan_list,
	      int sbmode, int buffers_sent, int delayed, int dmode)
{
    int     status;

    CHECK(status = check_arbrdy(hw, nchan, chan_list, dmode));
    if (nchan > 0)
    {
	if (buffers_sent == 0)
	{
	    if (status == 0)
	    {
		(void) fprintf(stderr,
			       "%s: no arbrdy before sending any data?\n",
			       progname);
		return -1;
	    }
	}

	if (buffers_sent == 1)
	{
	    if (sbmode == E1432_SRCBUFFER_CONTINUOUS ||
		sbmode == E1432_SRCBUFFER_PERIODIC_AB)
	    {
		if (status == 0)
		{
		    (void) fprintf(stderr,
				   "%s: no arbrdy after sending A data?\n",
				   progname);
		    return -1;
		}
	    }
	    else
	    {
		/* Oneshot or periodic_a */
		if (status != 0)
		{
		    (void) fprintf(stderr,
				   "%s: arbrdy after sending A data?\n",
				   progname);
		    return -1;
		}
	    }
	}

	if (buffers_sent >= 2)
	{
	    if (sbmode == E1432_SRCBUFFER_CONTINUOUS)
	    {
		if (delayed)
		{
		    if (status == 0)
		    {
			(void) fprintf(stderr,
				       "%s: no arbrdy after waiting?\n",
				       progname);
			return -1;
		    }
		}
		else
		{
		    /* Could be either way */
		}
	    }
	    else
	    {
		if (status != 0)
		{
		    (void) fprintf(stderr,
				   "%s: arbrdy after sending data?\n",
				   progname);
		    return -1;
		}
	    }
	}
    }

    return 0;
}

static int
wait_arbrdy(E1432ID hw, int chan, int dmode)
{
    clock_t start, timeout;
    int     status;

    timeout = 2 * CLOCKS_PER_SEC;
    start = clock();
    while ((status = e1432_check_src_arbrdy(hw, chan, dmode)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_check_src_arbrdy(hw, chan, dmode)) == 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_check_src_arbrdy: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_check_src_arbrdy: returned %d\n",
		       progname, status);
	return -1;
    }

    return 0;
}

/* Fill one source buffer, either A or B, for all source channels we
   are using. */
static int
xfer_src_data(E1432ID hw, int nchan, SHORTSIZ16 *chan_list,
	      LONGSIZ32 *buf, long xfersize, long sbsize, int dmode)
{
    long    xfer, offset, npoint;
    int     i;

    /* We must loop through all channels doing only xfersize words at
       a time. */
    offset = 0;
    npoint = sbsize;
    while (npoint > 0)
    {
	xfer = xfersize;
	if (xfer > npoint)
	    xfer = npoint;

	for (i = 0; i < nchan; i++)
	{
	    CHECK(wait_arbrdy(hw, chan_list[i], dmode));
	    CHECK(e1432_write_srcbuffer_data(hw, chan_list[i],
					     buf + offset, xfer, dmode));
	}

	npoint -= xfer;
	offset += xfer;
    }

    return 0;
}

static int
init(SHORTSIZ16 laddr, E1432ID *hw, int *group,
     int ch1, int ch2, int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig;
    int     limit;

    error_print = 1;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(1, &laddr, hw));
    CHECK(e1432_get_hwconfig(1, &laddr, &hwconfig));

    limit = ch2 ? 2 : 1;
    if (hwconfig.source_chans < limit)
    {
	(void) fprintf(stderr, "%s: Need %d source channel(s)\n",
		       progname, limit);
	return -1;
    }
    if (hwconfig.dram_size == 0)
    {
	(void) fprintf(stderr, "%s: Need source DRAM for this test\n",
		       progname);
	return -1;
    }

    *nchan = 0;
    if (ch1)
	chan_list[(*nchan)++] = E1432_SOURCE_CHAN(1);
    if (ch2)
	chan_list[(*nchan)++] = E1432_SOURCE_CHAN(2);

    *group = e1432_create_channel_group(*hw, *nchan, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

static int
setup(E1432ID hw, int group, int mode, int sbmode, long sbsize)
{
    CHECK(e1432_reset_measure(hw, group));

    CHECK(e1432_set_active(hw, group, E1432_CHANNEL_ON));
    CHECK(e1432_set_clock_freq(hw, group, 51200));
    CHECK(e1432_set_ramp_rate(hw, group, 0));
    CHECK(e1432_set_range(hw, group, 1));
    CHECK(e1432_set_span(hw, group, 20000));

    CHECK(e1432_set_source_mode(hw, group, mode));
    CHECK(e1432_set_srcbuffer_mode(hw, group, sbmode));
    CHECK(e1432_set_srcbuffer_size(hw, group, sbsize));

    CHECK(e1432_set_srcbuffer_init(hw, group, E1432_SRCBUFFER_INIT_EMPTY));

    return 0;
}

static char *
mode_to_str(int mode)
{
    switch (mode)
    {
    case E1432_SOURCE_MODE_ARB:
	return "Arb   ";
    case E1432_SOURCE_MODE_BARB:
	return "Barb  ";
    }
    return "Error!";
}

static char *
sbmode_to_str(int sbmode)
{
    switch (sbmode)
    {
    case E1432_SRCBUFFER_CONTINUOUS:
	return "Cont   ";
    case E1432_SRCBUFFER_PERIODIC_A:
	return "Per_A  ";
    case E1432_SRCBUFFER_PERIODIC_AB:
	return "Per_AB ";
    case E1432_SRCBUFFER_ONESHOT:
	return "Oneshot";
    }
    return "Error! ";
}

static int
good(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
     int mode, int sbmode, long sbsize, long xfersize,
     int verbose)
{
    LONGSIZ32 *buf;
    int     buffers_sent, dmode, overread_ok;

    if (verbose)
	(void) printf("good     (%s, %s, %6ld, %6ld)\n",
		      mode_to_str(mode), sbmode_to_str(sbmode),
		      sbsize, xfersize);

    buffers_sent = 0;
    if (sbmode == E1432_SRCBUFFER_CONTINUOUS ||
	sbmode == E1432_SRCBUFFER_PERIODIC_AB)
	dmode = E1432_SRC_DATA_MODE_WAITAB;
    else
	dmode = E1432_SRC_DATA_MODE_WAITA;

    overread_ok = sbmode == E1432_SRCBUFFER_CONTINUOUS;

    /* Set up the source */
    CHECK(setup(hw, group, mode, sbmode, sbsize));

    /* Allocate source buffer */
    buf = malloc(sbsize * sizeof *buf);
    if (buf == NULL)
    {
	(void) fprintf(stderr, "%s: malloc failed\n", progname);
	return -1;
    }

    /* Zero the data */
    (void) memset(buf, 0, sbsize * sizeof *buf);

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list, 0));
    CHECK(check_arbrdy2(hw, nchan, chan_list, sbmode, buffers_sent, 0, dmode));

    /* Fill A */
    if (verbose > 2)
	(void) printf("Xfer scan 1 (A buffer)\n");
    CHECK(xfer_src_data(hw, nchan, chan_list, buf, xfersize, sbsize, dmode));
    buffers_sent++;

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list, 0));
    CHECK(check_arbrdy2(hw, nchan, chan_list, sbmode, buffers_sent, 0, dmode));

    if (sbmode == E1432_SRCBUFFER_CONTINUOUS ||
	sbmode == E1432_SRCBUFFER_PERIODIC_AB)
    {
	if (verbose > 2)
	    (void) printf("Xfer scan 2 (B buffer)\n");
	CHECK(xfer_src_data(hw, nchan, chan_list, buf, xfersize,
			    sbsize, dmode));
	buffers_sent++;
    }

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list, 0));
    CHECK(check_arbrdy2(hw, nchan, chan_list, sbmode, buffers_sent, 0, dmode));

    (void) sleep(1);

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list, 0));
    CHECK(check_arbrdy2(hw, nchan, chan_list, sbmode, buffers_sent, 0, dmode));

    /* Start measurement */
    if (verbose > 2)
	(void) printf("e1432_init_measure\n");
    CHECK(e1432_init_measure(hw, group));

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list, overread_ok));
    CHECK(check_arbrdy2(hw, nchan, chan_list, sbmode, buffers_sent, 0, dmode));

    (void) sleep(1);

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list, overread_ok));
    CHECK(check_arbrdy2(hw, nchan, chan_list, sbmode, buffers_sent, 1, dmode));

    if (sbmode == E1432_SRCBUFFER_CONTINUOUS)
    {
	if (verbose > 2)
	    (void) printf("Xfer scan 3 (A buffer)\n");
	CHECK(xfer_src_data(hw, nchan, chan_list, buf, xfersize,
			    sbsize, dmode));
	buffers_sent++;
    }

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list, overread_ok));
    CHECK(check_arbrdy2(hw, nchan, chan_list, sbmode, buffers_sent, 0, dmode));

    (void) sleep(1);

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list, overread_ok));
    CHECK(check_arbrdy2(hw, nchan, chan_list, sbmode, buffers_sent, 1, dmode));

    free(buf);

    return 0;
}

static int
bad_xfer(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
	 int mode, int sbmode, long sbsize, long xfersize,
	 int verbose)
{
    LONGSIZ32 *buf;
    int     status, dmode;

    if (verbose)
	(void) printf("bad_xfer (%s, %s, %6ld, %6ld)\n",
		      mode_to_str(mode), sbmode_to_str(sbmode),
		      sbsize, xfersize);

    if (sbmode == E1432_SRCBUFFER_CONTINUOUS ||
	sbmode == E1432_SRCBUFFER_PERIODIC_AB)
	dmode = E1432_SRC_DATA_MODE_WAITAB;
    else
	dmode = E1432_SRC_DATA_MODE_WAITA;

    /* Set up the source */
    CHECK(setup(hw, group, mode, sbmode, sbsize));

    /* Allocate source buffer */
    buf = malloc(sbsize * sizeof *buf);
    if (buf == NULL)
    {
	(void) fprintf(stderr, "%s: malloc failed\n", progname);
	return -1;
    }

    /* Zero the data */
    (void) memset(buf, 0, sbsize * sizeof *buf);

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list, 0));
    CHECK(check_arbrdy2(hw, nchan, chan_list, sbmode, 0, 0, dmode));

    /* Fill A */
    if (verbose > 2)
	(void) printf("Xfer scan 1 (A buffer)\n");
    CHECK(e1432_print_errors(0));
    error_print = 0;
    status = xfer_src_data(hw, nchan, chan_list, buf, xfersize,
			   sbsize, dmode);
    error_print = 1;
    CHECK(e1432_print_errors(1));
    if (status == 0)
    {
	(void) fprintf(stderr,
		       "%s: no xfer failure with mode %s, sbmode %s, sbsize %ld, xfersize %ld\n",
		       progname, mode_to_str(mode),
		       sbmode_to_str(sbmode), sbsize, xfersize);
	return -1;
    }

    free(buf);

    return 0;
}

static int
bad_setup(E1432ID hw, int group,
	  int mode, int sbmode, long sbsize, int verbose)
{
    int     status;

    if (verbose)
	(void) printf("bad_setup(%s, %s, %6ld)\n",
		      mode_to_str(mode), sbmode_to_str(sbmode),
		      sbsize);

    CHECK(e1432_print_errors(0));
    error_print = 0;
    status = setup(hw, group, mode, sbmode, sbsize);
    error_print = 1;
    CHECK(e1432_print_errors(1));
    if (status == 0)
    {
	(void) fprintf(stderr,
		       "%s: no failure with mode %s, sbmode %s, sbsize %ld\n",
		       progname, mode_to_str(mode),
		       sbmode_to_str(sbmode), sbsize);
	return -1;
    }
    return 0;
}

static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list, int verbose)
{
    int     sbmax;

    if (nchan < 2 && chan_list[0] == E1432_SOURCE_CHAN(1))
	sbmax = 40960;
    else
	sbmax = 20480;

    /* Burst arb, DRAM never used
       Only periodic modes should work
       Xfer size limited to 4096
       Buffer size limited to sbmax */
    CHECK(bad_setup(hw, group, E1432_SOURCE_MODE_BARB,
		    E1432_SRCBUFFER_CONTINUOUS, 20480, verbose));
    CHECK(bad_setup(hw, group, E1432_SOURCE_MODE_BARB,
		    E1432_SRCBUFFER_ONESHOT, 20480, verbose));
    CHECK(bad_setup(hw, group, E1432_SOURCE_MODE_BARB,
		    E1432_SRCBUFFER_PERIODIC_A, sbmax + 1, verbose));
    CHECK(bad_setup(hw, group, E1432_SOURCE_MODE_BARB,
		    E1432_SRCBUFFER_PERIODIC_AB, sbmax + 1, verbose));

    CHECK(bad_xfer(hw, group, nchan, chan_list, E1432_SOURCE_MODE_BARB,
		   E1432_SRCBUFFER_PERIODIC_A, sbmax, 4097, verbose));
    CHECK(bad_xfer(hw, group, nchan, chan_list, E1432_SOURCE_MODE_BARB,
		   E1432_SRCBUFFER_PERIODIC_AB, sbmax, 4097, verbose));

    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_BARB,
	       E1432_SRCBUFFER_PERIODIC_A, 1021, 4096, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_BARB,
	       E1432_SRCBUFFER_PERIODIC_AB, 1021, 4096, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_BARB,
	       E1432_SRCBUFFER_PERIODIC_A, 4096, 4096, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_BARB,
	       E1432_SRCBUFFER_PERIODIC_AB, 4096, 4096, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_BARB,
	       E1432_SRCBUFFER_PERIODIC_A, sbmax, 4096, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_BARB,
	       E1432_SRCBUFFER_PERIODIC_AB, sbmax, 4096, verbose));

    /* Arb, should allow all xfer modes and sizes, but large source
       buffers should work only for periodic_a and oneshot */
    CHECK(bad_setup(hw, group, E1432_SOURCE_MODE_ARB,
		    E1432_SRCBUFFER_CONTINUOUS, sbmax + 1, verbose));
    CHECK(bad_setup(hw, group, E1432_SOURCE_MODE_ARB,
		    E1432_SRCBUFFER_PERIODIC_AB, sbmax + 1, verbose));

    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_CONTINUOUS, 4096, 4096, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_PERIODIC_AB, 4096, 4096, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_CONTINUOUS, sbmax, 4096, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_PERIODIC_AB, sbmax, 4096, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_CONTINUOUS, sbmax, sbmax, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_PERIODIC_AB, sbmax, sbmax, verbose));

    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_PERIODIC_A, 4096, 4096, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_ONESHOT, 4096, 4096, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_PERIODIC_A, sbmax, 4096, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_ONESHOT, sbmax, 4096, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_PERIODIC_A, sbmax, sbmax, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_ONESHOT, sbmax, sbmax, verbose));

    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_PERIODIC_A, 5 * sbmax, 4096, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_ONESHOT, 5 * sbmax, 4096, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_PERIODIC_A, 5 * sbmax, 5 * sbmax, verbose));
    CHECK(good(hw, group, nchan, chan_list, E1432_SOURCE_MODE_ARB,
	       E1432_SRCBUFFER_ONESHOT, 5 * sbmax, 5 * sbmax, verbose));

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-12uvV] [-L laddr]\n"
		   "\t-1: Use source channel 1\n"
		   "\t-2: Use source channel 2\n"
		   "\t-L: Logical address is <laddr>, default 8\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr;
    SHORTSIZ16 chan_list[2];
    char   *p;
    int     c, ch1, ch2, verbose, group, nchan;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    ch1 = 0;
    ch2 = 0;
    laddr = 8;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "12L:uvV")) != -1)
	switch (c)
	{
	case '1':
	    ch1 = 1;
	    break;
	case '2':
	    ch2 = 1;
	    break;
	case 'L':
	    laddr = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr < 0 || laddr > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    if (!ch1 && !ch2)
    {
	(void) fprintf(stderr, "%s: activate at least one source channel\n",
		       progname);
	usage();
    }

    /* Run the measurement */
    if (init(laddr, &hw, &group, ch1, ch2, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
